(function () {
    let app = angular.module('products', []);

    app.directive('productsOverlay', function () {
        return {
            restrict: 'E',
            scope: {},
            templateUrl: 'templates/products/overlay.html',
            controller: function ($scope, $rootScope) {
                let ctrl = this;
                $scope.GetLocalized = Helpers.GetLocalized;

                $scope.$on(GameEvents.ProductChange, () => {
                    ctrl.products = $rootScope.settings.products.map(product => {
                        let productType = ProductTypes.find(x => x.name == product.productTypeName);
                        let totalUsers = _.sum(_.toArray(product.users));
                        return {
                            product: product,
                            totalUsers: totalUsers,
                            incomePerDay: productType.incomePerUserPerDay * totalUsers
                        }
                    });
                });
                $scope.$broadcast(GameEvents.ProductChange);

                ctrl.showProduct = product => {
                    $rootScope.setActiveView('product');
                    $rootScope.selectedProductId = product.id;
                };

                ctrl.createProduct = () => {
                    $rootScope.selectedProductId = null;
                    $rootScope.setActiveView('product');
                }
            },
            controllerAs: 'ctrl'
        }
    });

    app.directive('productCreate', function () {
        return {
            restrict: 'E',
            templateUrl: 'templates/products/create.html',
            controller: function ($scope, $rootScope, $timeout) {
                let ctrl = this;
                $scope.GetLocalized = Helpers.GetLocalized;

                ctrl.platformNames = _.toArray(Enums.PlatformNames);
                ctrl.product = $rootScope.settings.products.find(x => x.id == $rootScope.selectedProductId);
                ctrl.isUpgrade = false;
                ctrl.view = ctrl.product != null && ctrl.product.underDevelopment ? 'implementation' : 'details';

                ctrl.cancelCreate = () => {
                    if (ctrl.isUpgrade) {
                        $scope.$parent.ctrl.tab = 'stats';
                    } else {
                        $rootScope.closeAllUi();
                    }
                };

                if (ctrl.product == null) {
                    ctrl.product = {
                        id: chance.guid(),
                        name: '',
                        ageInDays: 0,
                        version: 0,
                        stats: [],
                        completed: false,
                        features: [],
                        users: {Desktop: 0, Mobile: 0, Web: 0},
                        underDevelopment: true,
                        hype: 0,
                        servers: {},
                        userSegments: [],
                        reviews: [],
                        activeMarketingPackages: []
                    };
                } else {
                    ctrl.isUpgrade = true;
                }

                let loadPieCharts = (selector, labels, series) => {
                    let data = {
                        labels: labels,
                        series: series
                    };

                    let options = {
                        width: '294px',
                        height: '155px',
                        labelInterpolationFnc: function (label, index) {
                            return label + " " + data.series[index] + '%';
                        },
                        chartPadding: 0,
                        labelOffset: 10,
                        labelDirection: 'explode',
                    };

                    new Chartist.Pie(selector, data, options);
                };

                $scope.$watch('createCtrl.view', view => {
                    if (view == 'details') {
                        $timeout(() => {
                            ProductTypes.forEach(productType => {
                                let series = _.toArray(productType.users);
                                let sum = _.sum(series);
                                let percentages = series.map(value => {
                                    return Math.round(value * 100 / sum);
                                });

                                loadPieCharts(`#chart-${productType.name}`,
                                    Object.keys(productType.users).map(x => Helpers.GetLocalized(x)),
                                    percentages
                                );
                            });
                        }, 150);
                    }
                });

                // Initialize variables
                let researchFrameworks = $rootScope.settings.researchInventory
                    .filter(x => x.frameworkName != null && Helpers.CalculateRequiredResearchPoints(null, x.frameworkName) <= x.researchPoints)
                    .map(x => x.frameworkName);

                ctrl.frameworks =
                    _.orderBy(researchFrameworks
                        .map(frameworkName => {
                            let result = Helpers.Clone(Frameworks.find(x => x.name == frameworkName));
                            result.friendlyName = Helpers.GetLocalized(result.name);
                            return result;
                        }), 'order');

                ctrl.framework = ctrl.isUpgrade
                    ? ctrl.frameworks.find(x => x.name == ctrl.product.frameworkName)
                    : ctrl.frameworks[0];

                ctrl.productTypes = ProductTypes;

                ctrl.selectProductType = productType => {
                    if (!ctrl.isUpgrade) {
                        ctrl.product.productTypeName = productType.name;
                    } else {
                        $rootScope.showMessage(``, `You cannot change product type when upgrading.`)
                    }
                };

                ctrl.isCreatePageOneValid = () => {
                    let nameValid = ctrl.product.name.length > 1 && !$rootScope.settings.products.map(x => x.name).includes(ctrl.product.name);
                    let productTypeValid = ctrl.product.productTypeName != null;
                    let frameworkSelected = ctrl.framework != null && ctrl.framework.id != 0;
                    let frameworkSupported = ctrl.framework.maxFeatures >= ctrl.product.features.length;

                    if ($scope.ctrl.product == null) {
                        return nameValid && productTypeValid && frameworkSelected;
                    } else { // Upgrade
                        return frameworkSelected && frameworkSupported;
                    }
                };

                ctrl.isCreatePageTwoValid = () => {
                    return ctrl.features.filter(x => x.selected).length > 0;
                };

                ctrl.showFeatures = () => {
                    let researchFeatures = $rootScope.settings.researchInventory
                        .filter(x => x.featureName != null && Helpers.CalculateRequiredResearchPoints(x.featureName) <= x.researchPoints)
                        .map(x => x.featureName);

                    ctrl.features = _.flatten(ctrl.platformNames.map(platformName => {

                        return researchFeatures.map(featureName => {
                            return {
                                featureName: featureName,
                                platformName: platformName,
                                selected: ctrl.product.features.some(x => x.featureName == featureName && x.platformName == platformName)
                            };
                        });
                    }));

                    $scope.$watch('createCtrl.features', () => {
                        ctrl.featuresLeft = ctrl.framework.maxFeatures - ctrl.features.filter(x => x.selected).length;
                    }, true);

                    ctrl.view = 'features';
                };

                ctrl.toggleFeature = featureInfo => {
                    if (ctrl.featuresLeft <= 0 && !featureInfo.selected) {
                        return;
                    }

                    let existingFeature = ctrl.product.features.find(x => x.featureName == featureInfo.featureName && x.platformName == featureInfo.platformName);

                    // Feature already exists in product
                    if (existingFeature && featureInfo.selected) {
                        $rootScope.confirm(``, Helpers.GetLocalized('confirm_disable_feature'), () => {
                            featureInfo.selected = false;
                        });
                    } else {
                        featureInfo.selected = !featureInfo.selected;

                    }
                };

                let loadMissingFeatures = () => {
                    let totalCu = Helpers.CalculateTotalCuByProduct(ctrl.product);
                    let requiredCu = Helpers.CalculateRequiredCuForProduct(ctrl.product);
                    let freeCu = totalCu - requiredCu;


                    ctrl.missingFeatures = ctrl.product.features.filter(x => x.level == 0).map(featureInfo => {
                        let feature = Features.find(x => x.name == featureInfo.featureName);
                        let stacks = Helpers.ConvertRequirementsIntoStacks(feature.requirements);
                        return {
                            featureInfo: featureInfo,
                            stacks: stacks,
                            available: !stacks.some(x => !x.isAvailableInInventory),
                            cuAvailable: freeCu > feature.requiredComputeUnit
                        };
                    });
                };

                $scope.$watch('createCtrl.product', product => {
                    if (product == null) return;
                    loadMissingFeatures();
                }, true);

                Helpers.RootScopeWatchCollectionAndDestroy($scope, 'settings.inventory', () => {
                    if (ctrl.product != null)
                        loadMissingFeatures();
                });

                ctrl.confirmVersion = () => {
                    ctrl.product.newFrameworkName = ctrl.framework.name;
                    if(ctrl.product.frameworkName == null) {
                        ctrl.product.frameworkName = ctrl.product.newFrameworkName;
                    }

                    ctrl.features.forEach(featureInfo => {
                        let existingFeature = ctrl.product.features.find(f => f.featureName == featureInfo.featureName && f.platformName == featureInfo.platformName);

                        if (existingFeature == null && featureInfo.selected) {
                            ctrl.product.features.push({
                                featureName: featureInfo.featureName,
                                platformName: featureInfo.platformName,
                                level: 0,
                            });
                        }

                        // Remove deselected but existing features
                        if (existingFeature != null && !featureInfo.selected) {
                            _.remove(ctrl.product.features, x => x.featureName == featureInfo.featureName && x.platformName == featureInfo.platformName);
                            _.remove(ctrl.product.userSegments, x => x.featureName == featureInfo.featureName && x.platformName == featureInfo.platformName);
                        }
                    });

                    if (!ctrl.isUpgrade) {
                        $rootScope.settings.products.push(ctrl.product);
                        wallhack.sendEvent("created_new_product", ctrl.product.name);
                        $rootScope.$broadcast(GameEvents.ProductChange);
                    }

                    ctrl.product.underDevelopment = true;
                    $scope.$parent.ctrl.product = ctrl.product;

                    ctrl.view = 'implementation';
                };

                ctrl.implementFeature = featureInfo => {
                    // Grab components
                    let feature = Features.find(x => x.name == featureInfo.featureInfo.featureName);

                    // Grab components
                    Object.keys(feature.requirements).forEach(componentName => {
                        $rootScope.settings.inventory[componentName] -= feature.requirements[componentName];
                    });

                    featureInfo.featureInfo.level++;
                };

                ctrl.cancelVersion = () => {
                    $rootScope.confirm(``, Helpers.GetLocalized('cancel_version_confirm'), () => {
                        if(ctrl.product.version == 0) {
                            _.remove($rootScope.settings.products, x => x.id == ctrl.product.id);
                            $rootScope.closeAllUi();
                        } else {
                            ctrl.product.newFrameworkName = null;
                            ctrl.product.underDevelopment = false;
                            ctrl.view = 'details';
                        }

                        $rootScope.$broadcast(GameEvents.ProductChange);
                    });
                };

                ctrl.releaseVersion = () => {
                    let productType = ProductTypes.find(x => x.name == ctrl.product.productTypeName);

                    let noneMatchingFeatures = [];
                    let successfulFeatures = [];

                    ctrl.product.frameworkName = ctrl.product.newFrameworkName;
                    ctrl.product.newFrameworkName  = null;

                    // Find matching features
                    ctrl.product.features.forEach(featureInfo => {
                        if (productType.features.includes(featureInfo.featureName)) { // matches productType
                            successfulFeatures.push(featureInfo.featureName);
                        } else {
                            noneMatchingFeatures.push(featureInfo.featureName);
                        }

                        // Add new user segment for new features
                        if (featureInfo.level == 1) {
                            ctrl.product.userSegments.push(Helpers.GenerateUserSegment(
                                featureInfo.featureName,
                                featureInfo.platformName,
                                ctrl.product.productTypeName));

                            ctrl.product.hype = Math.min(ctrl.product.hype + 5, 100);
                        }
                    });

                    if (ctrl.product.stats.length == 0) {
                        ctrl.product.stats.push({
                            day: GetDateDiffInDays($rootScope.settings.date),
                            gainedUsers: 0,
                            totalUsers: 0,
                            income: 0
                        });
                    }

                    ctrl.product.completed = true;
                    ctrl.product.version++;
                    ctrl.product.underDevelopment = false;

                    // Create reviews
                    ctrl.product.reviews = [
                        ...successfulFeatures.map(x => {
                            return {featureName: x, positive: true};
                        }),
                        ...noneMatchingFeatures.map(x => {
                            return {featureName: x, positive: false};
                        }),
                    ].map(feature => {
                        return {
                            gender: _.random(0, 1) == 0 ? "male" : "female",
                            positive: feature.positive,
                            message: feature.positive
                                ? `${ feature.featureName } is a great fit for ${ ctrl.product.name }!`
                                : `I don't understand the purpose of the new ${ feature.featureName }.`
                        }
                    });

                    $scope.$parent.ctrl.tab = 'stats';
                    $rootScope.selectedProductId = ctrl.product.id;
                }
            },
            controllerAs: 'createCtrl'
        }
    });

    app.directive('product', function () {
        return {
            restrict: 'E',
            scope: {},
            templateUrl: 'templates/products/index.html',
            controller: function ($scope, $rootScope) {
                let ctrl = this;
                $scope.GetLocalized = Helpers.GetLocalized;
                ctrl.product = $rootScope.settings.products.find(x => x.id == $rootScope.selectedProductId);
                if (ctrl.product != null && ctrl.product.completed) {
                    ctrl.tab = 'stats';
                } else {
                    ctrl.tab = 'create';
                }

                ctrl.deleteProduct = product => {
                    $rootScope.confirm(``, "Are you sure?", () => {
                        _.remove($rootScope.settings.products, p => p.id == product.id);
                        $rootScope.$broadcast(GameEvents.ProductChange);
                        $rootScope.closeAllUi();
                    });
                };
            },
            controllerAs: 'ctrl'
        }
    });

    app.directive('productStats', function () {
        return {
            restrict: 'E',
            scope: {},
            templateUrl: 'templates/products/stats.html',
            controller: function ($scope, $rootScope, $timeout) {
                let ctrl = this;
                $scope.GetLocalized = Helpers.GetLocalized;
                ctrl.product = $scope.$parent.ctrl.product;
                ctrl.view = '';

                ctrl.hostingExpenses = Helpers.CalculateHostingExpenses(ctrl.product);
                ctrl.underDevelopment = ctrl.product.newFrameworkName != null;

                $scope.$watch('statsCtrl.product.features', () => {
                    let framework = Frameworks.find(x => x.name == ctrl.product.frameworkName);
                    ctrl.newVersionAvailable = ctrl.product != null ?
                        !ctrl.product.features.some(x => x.level < framework.maxFeatureLevel)
                        : false;
                }, true);

                $scope.$watchCollection('statsCtrl.product.stats', stats => {
                    $timeout(loadChart, 300);
                    ctrl.totalIncome = _.sum(stats.map(x => x.income));
                });

                $scope.$watch('statsCtrl.product.userSegments', userSegments => {
                    ctrl.usersLeft = {
                        Desktop: _.sumBy(userSegments.filter(x => x.platformName == 'Desktop'), 'usersLeft'),
                        Mobile: _.sumBy(userSegments.filter(x => x.platformName == 'Mobile'), 'usersLeft'),
                        Web: _.sumBy(userSegments.filter(x => x.platformName == 'Web'), 'usersLeft'),
                    }
                }, true);

                ctrl.showVersionUpgrade = product => {
                    $scope.$parent.ctrl.tab = 'create';
                };

                let loadChart = () => {
                    let highest = _.max(ctrl.product.stats.map(x => x.income));
                    ctrl.productStats = Helpers.Clone(_.takeRight(ctrl.product.stats, 10)).map(stat => {
                        stat.percentage = stat.income * 100 / highest;
                        stat.income = Math.round(stat.income);
                        stat.gainedUsers = Math.round(stat.gainedUsers);
                        return stat;
                    });

                    let labels = ctrl.productStats.map(x => x.day);
                    let series = ctrl.productStats.map(x => x.totalUsers || 0);

                    let options = {
                        width: '760px',
                        height: '160px',
                        showArea: true,
                        fullWidth: true,
                        axisX: {
                            labelOffset: { x: -10 },
                        },
                        axisY: {
                            labelOffset: { y: 5 },
                            labelInterpolationFnc: function (value) {
                                if(value > 1000) {
                                    return (value / 1000) + 'K';
                                }
                            }
                        }
                    };

                    new Chartist.Line('.ct-chart', {
                        labels: labels,
                        series: [series]
                    }, options);
                }
            },
            controllerAs: 'statsCtrl'
        }
    });

    app.directive('productUpgrades', function () {
        return {
            restrict: 'E',
            scope: {},
            templateUrl: 'templates/products/upgrades.html',
            controller: function ($scope, $rootScope) {
                let ctrl = this;
                $scope.GetLocalized = Helpers.GetLocalized;
                ctrl.product = $scope.$parent.ctrl.product;

                // Features
                let loadFeatures = () => {
                    if (ctrl.product == null) return;

                    let framework = Frameworks.find(x => x.name == ctrl.product.frameworkName);
                    let totalCu = Helpers.CalculateTotalCuByProduct(ctrl.product);
                    let requiredCu = Helpers.CalculateRequiredCuForProduct(ctrl.product);
                    let freeCu = totalCu - requiredCu;

                    ctrl.featureResults = ctrl.product.features.filter(x => x.level != 0).map(featureInfo => {
                        let result = {
                            featureInfo: featureInfo,
                            feature: Helpers.Clone(Features.find(x => x.name == featureInfo.featureName))
                        };

                        result.levels = _.range(framework.maxFeatureLevel).map(x => {
                            return {level: x + 1, enabled: featureInfo.level >= (x + 1)}
                        });

                        result.stacks = _.orderBy(Helpers.ConvertRequirementsIntoStacks(Helpers.GetRequirementsForFeatureLevel(result.feature, featureInfo.level+1)), x => [x.component.type, x.component.name]);
                        result.productionHours = Math.round(_.sum(Object.keys(result.feature.requirements).map(componentName => Helpers.CalculateComponentProductionHours(Components.find(x => x.name == componentName)) * result.feature.requirements[componentName])));
                        result.upgradeAvailable = framework.maxFeatureLevel > featureInfo.level && !result.stacks.some(x => !x.isAvailableInInventory);
                        result.nextLevelUnlock = Helpers.GenerateUserSegment(result.feature.name, result.featureInfo.platformName, ctrl.product.productTypeName);
                        result.cuAvailable = freeCu >= result.feature.requiredComputeUnit;

                        return result;
                    });
                };

                Helpers.RootScopeWatchCollectionAndDestroy($scope, 'settings.inventory', () => loadFeatures());

                ctrl.upgradeFeature = featureResult => {
                    // Grab components
                    Object.keys(featureResult.feature.requirements).forEach(componentName => {
                        $rootScope.settings.inventory[componentName] -= featureResult.feature.requirements[componentName];
                    });

                    ctrl.product.userSegments.push(Helpers.GenerateUserSegment(
                        featureResult.featureInfo.featureName,
                        featureResult.featureInfo.platformName,
                        ctrl.product.productTypeName
                    ));

                    featureResult.featureInfo.users = 3000;
                    featureResult.featureInfo.level++;
                    ctrl.product.hype = Math.min(ctrl.product.hype + featureResult.feature.hype, 100);
                    wallhack.sendEvent("upgrade_feature", {
                        productName: ctrl.product.name,
                        featureName: featureResult.featureInfo.featureName,
                        platformName: featureResult.featureInfo.platformName,
                        newLevel: featureResult.featureInfo.level,
                        hype: Math.round(ctrl.product.hype)
                    });
                };
            },
            controllerAs: 'upgradesCtrl'
        }
    });

    app.directive('productHosting', function () {
        return {
            restrict: 'E',
            scope: {},
            templateUrl: 'templates/products/hosting.html',
            controller: function ($scope, $rootScope) {
                let ctrl = this;
                $scope.GetLocalized = Helpers.GetLocalized;
                ctrl.product = $scope.$parent.ctrl.product;

                let loadServers = () => {
                    ctrl.servers = Servers.map(server => {
                        let result = Helpers.Clone(server);
                        result.stacks = _.orderBy(Helpers.ConvertRequirementsIntoStacks(result.requirements), x => [x.component.type, x.component.name]);
                        result.isAvailable = !result.stacks.some(x => !x.isAvailableInInventory);
                        result.instances = ctrl.product.servers[server.name] || 0;
                        return result;
                    });

                    ctrl.totalCu = Helpers.CalculateTotalCuByProduct(ctrl.product);
                    ctrl.requiredCu = Helpers.CalculateRequiredCuForProduct(ctrl.product);
                    ctrl.hostingPercentage = Math.min(Math.round(ctrl.requiredCu * 100 / ctrl.totalCu), 100);
                };

                Helpers.RootScopeWatchCollectionAndDestroy($scope, 'settings.inventory', () => loadServers());

                ctrl.addInstance = server => {
                    ctrl.product.servers[server.name] = (ctrl.product.servers[server.name] || 0) + 1;

                    // Grab components
                    Object.keys(server.requirements).forEach(componentName => {
                        $rootScope.settings.inventory[componentName] -= server.requirements[componentName];
                    });

                    loadServers();
                };
            },
            controllerAs: 'hostingCtrl'
        }
    });

    app.directive('productMarketing', function () {
        return {
            restrict: 'E',
            scope: {},
            templateUrl: 'templates/products/marketing.html',
            controller: function ($scope, $rootScope) {
                let ctrl = this;
                $scope.GetLocalized = Helpers.GetLocalized;
                ctrl.product = $scope.$parent.ctrl.product;

                // Marketing Packages
                let loadMarketingPackages = () => {
                    if (ctrl.product == null) return;

                    ctrl.marketingPackagesResults = MarketingPackages.map(marketingPackage => {
                        let result = Helpers.Clone(marketingPackage);
                        let activePackage = ctrl.product.activeMarketingPackages.find(x => x.marketingPackageName == result.name);
                        result.stacks = _.orderBy(Helpers.ConvertRequirementsIntoStacks(result.requirements), x => [x.component.type, x.component.name]);
                        result.available = !result.stacks.some(x => !x.isAvailableInInventory);
                        result.hours = Math.round(result.durationInMinutes / 60);
                        result.activePackage = activePackage;
                        result.hoursLeft = activePackage != null ? Math.round(result.durationInMinutes / 60) : 0;
                        result.activePackageProgress = activePackage != null ? Math.floor(activePackage.completedMinutes * 100 / activePackage.totalMinutes) : 0;
                        return result;
                    });
                };

                Helpers.RootScopeWatchCollectionAndDestroy($scope, 'settings.inventory', () => loadMarketingPackages());

                ctrl.startMarketingCampaign = result => {
                    $rootScope.safeBuy(() => {
                        // Grab components
                        Object.keys(result.requirements).forEach(componentName => {
                            $rootScope.settings.inventory[componentName] -= result.requirements[componentName];
                        });

                        ctrl.product.activeMarketingPackages.push({
                            marketingPackageName: result.name,
                            totalMinutes: result.durationInMinutes,
                            completedMinutes: 0,
                            hype: result.hype
                        });
                        ctrl.product.hype = Math.min(ctrl.product.hype + result.hype, 100);

                        loadMarketingPackages();
                    }, result.price, Helpers.GetLocalized(result.name));
                };

                $scope.$watch('marketingCtrl.product.activeMarketingPackages', () => loadMarketingPackages(), true);
            },
            controllerAs: 'marketingCtrl'
        }
    });
})();
